<!DOCTYPE html>
<html lang="en">
<head>
    <title>Manage Subscription</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <style>
        .loading {
            display: none;
            width: 20px;
            height: 20px;
            border: 3px solid #f3f3f3;
            border-top: 3px solid #3498db;
            border-radius: 50%;
            animation: spin 1s linear infinite;
            margin-left: 10px;
        }
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
    </style>
    <script src="https://checkout.razorpay.com/v1/checkout.js"></script>
</head>
<body class="font-sans bg-gray-100">
    <div class="container p-6 mx-auto">
        <h1 class="mb-6 text-3xl font-bold text-gray-800">Manage Subscription</h1>
        @if (session('error'))
            <div class="px-4 py-3 mb-4 text-red-700 bg-red-100 border border-red-400 rounded" role="alert">
                <p>{{ session('error') }}</p>
            </div>
        @endif
        <div id="upgradeForm" class="p-6 bg-white rounded-lg shadow-lg">
            <div class="mb-4">
                <label for="planSelect" class="block mb-2 text-sm font-bold text-gray-700">Select a Plan</label>
                <select id="planSelect" class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                    <option value="">Choose an action</option>
                    @if (!$subscription)
                        <option value="new">Subscribe to a New Plan</option>
                    @else
                        <option value="upgrade">Upgrade Current Plan</option>
                    @endif
                    @foreach ($plans as $plan)
                        <option value="{{ $plan->Plan_id }}" data-price="{{ $plan->Price }}" data-currency="{{ $plan->currency->Currency_code ?? 'USD' }}">
                            {{ $plan->Name }} ({{ $plan->Price }} {{ $plan->currency->Currency_code ?? 'USD' }})
                        </option>
                    @endforeach
                </select>
            </div>
            <input type="hidden" id="companyId" value="{{ $companyProfile->Company_id ?? '' }}">
            <input type="hidden" id="subscriptionId" value="{{ $subscription->Subscription_id ?? '' }}">
            <input type="hidden" id="currentUrl" value="{{ url()->current() }}">
            <button id="subscribeButton" class="flex items-center px-4 py-2 text-white bg-blue-600 rounded-lg hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 disabled:bg-gray-400 disabled:cursor-not-allowed" disabled>
                {{ !$subscription ? 'Subscribe Now' : 'Upgrade Now' }}
                <span id="loadingSpinner" class="ml-2 loading"></span>
            </button>
            <div id="result" class="mt-4 text-sm"></div>
        </div>
    </div>

    <script>
        (function() {
            const planSelect = document.getElementById('planSelect');
            const subscribeButton = document.getElementById('subscribeButton');
            const loadingSpinner = document.getElementById('loadingSpinner');
            const resultDiv = document.getElementById('result');
            const companyId = document.getElementById('companyId').value;
            const subscriptionId = document.getElementById('subscriptionId').value;
            const currentUrl = document.getElementById('currentUrl').value;

            if (!planSelect || !subscribeButton || !loadingSpinner || !resultDiv) {
                console.error('One or more DOM elements not found');
                return;
            }

            planSelect.addEventListener('change', () => {
                const isNew = planSelect.value === 'new';
                subscribeButton.disabled = !planSelect.value || (isNew && !['new'].includes(planSelect.value)) || (!isNew && !planSelect.value.match(/^\d+$/));
                subscribeButton.textContent = isNew ? 'Subscribe Now' : 'Upgrade Now';
            });

            subscribeButton.addEventListener('click', () => {
                const planId = planSelect.value;
                const price = planSelect.options[planSelect.selectedIndex].getAttribute('data-price');
                const currency = planSelect.options[planSelect.selectedIndex].getAttribute('data-currency');
                const isNew = planId === 'new';

                if ((!isNew && !planId) || !companyId) {
                    resultDiv.innerHTML = '<p class="text-red-500">Please select a plan or ensure company is valid.</p>';
                    return;
                }

                subscribeButton.disabled = true;
                loadingSpinner.style.display = 'inline-block';

                const url = isNew ? '/api/subscriptions' : '/api/subscriptions/upgrade';
                const data = {
                    Company_id: companyId,
                    ...(isNew ? {} : { Subscription_id: subscriptionId }),
                    ...(isNew ? { Plan_id: planSelect.value.replace('new', planSelect.options[planSelect.selectedIndex + 1].value) } : { New_Plan_id: planId }),
                    Amount: price,
                    Payment_gateway: 'razorpay',
                    Currency: currency,
                    Start_date: new Date().toISOString().split('T')[0],
                    ...(isNew ? { End_date: new Date(new Date().setMonth(new Date().getMonth() + 1)).toISOString().split('T')[0] } : {}),
                    Status: 'pending'
                };

                fetch(url, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': '{{ csrf_token() }}'
                    },
                    body: JSON.stringify(data)
                })
                .then(response => response.json())
                .then(data => {
                    loadingSpinner.style.display = 'none';
                    if (data.status === 'pending') {
                        resultDiv.innerHTML = `<p class="text-yellow-500">${data.message}</p>`;
                        const options = {
                            key: data.data.key,
                            amount: data.data.amount,
                            currency: data.data.currency,
                            name: "{{ $companyProfile->company_name ?? 'Unknown Company' }}",
                            description: data.data.description,
                            order_id: data.data.order_id,
                            handler: function (response) {
                                fetch('/api/subscriptions/callback/razorpay', {
                                    method: 'POST',
                                    headers: {
                                        'Content-Type': 'application/json',
                                        'X-CSRF-TOKEN': '{{ csrf_token() }}'
                                    },
                                    body: JSON.stringify({
                                        Transaction_id: data.data.order_id,
                                        razorpay_payment_id: response.razorpay_payment_id,
                                        razorpay_order_id: response.razorpay_order_id,
                                        razorpay_signature: response.razorpay_signature,
                                        ...(isNew ? { Plan_id: planSelect.value.replace('new', planSelect.options[planSelect.selectedIndex + 1].value) } : { New_Plan_id: planId })
                                    })
                                })
                                .then(cbResponse => cbResponse.json())
                                .then(cbData => {
                                    if (cbData.status === 'success') {
                                        resultDiv.innerHTML = `<p class="text-green-500">${cbData.message}</p>`;
                                        setTimeout(() => {
                                            window.location.href = currentUrl;
                                        }, 2000);
                                    } else {
                                        resultDiv.innerHTML = `<p class="text-red-500">Error: ${cbData.message}</p>`;
                                    }
                                })
                                .catch(cbError => {
                                    resultDiv.innerHTML = `<p class="text-red-500">Error: ${cbError.message}</p>`;
                                });
                            },
                            prefill: {
                                name: "{{ $companyProfile->company_name ?? 'Unknown Company' }}",
                                email: '',
                                contact: ''
                            },
                            notes: {
                                company_id: companyId,
                                ...(isNew ? { plan_id: planSelect.value.replace('new', planSelect.options[planSelect.selectedIndex + 1].value) } : { new_plan_id: planId })
                            },
                            theme: {
                                color: '#3399cc'
                            }
                        };
                        const rzp = new Razorpay(options);
                        rzp.open();
                        rzp.on('payment.failed', function (response) {
                            resultDiv.innerHTML = `<p class="text-red-500">Payment failed: ${response.error.description}</p>`;
                        });
                    } else if (data.status === 'success') {
                        resultDiv.innerHTML = `<p class="text-green-500">${data.message}</p>`;
                        setTimeout(() => {
                            window.location.href = currentUrl;
                        }, 2000);
                    } else {
                        resultDiv.innerHTML = `<p class="text-red-500">Error: ${data.message || 'Subscription failed'}</p>`;
                    }
                })
                .catch(error => {
                    loadingSpinner.style.display = 'none';
                    resultDiv.innerHTML = `<p class="text-red-500">Error: ${error.message}</p>`;
                })
                .finally(() => {
                    subscribeButton.disabled = false;
                });
            });
        })();
    </script>
</body>
</html>
